﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Inet.Viewer.Data;
using Inet.Viewer.Resources;
using System.Threading;

namespace Inet.Viewer.WinForms.Prompt
{
    /// <summary>
    /// A panel for a time range prompt field.
    /// </summary>
    public partial class TimeRangePromptField : PromptControl
    {
        /// <summary>
        /// Creates the panel.
        /// </summary>
        /// <param name="promptData">prompt field this panel is based one</param>
        /// <param name="validationDelegate">validation delegate for checking prompt values</param>
        /// <param name="hideNoValue">flag indicating the no value checkbox should be hidden</param>
        public TimeRangePromptField(PromptData promptData, ValidationDelegate validationDelegate, bool hideNoValue)
        {
            this.PromptData = promptData;
            InitializeComponent();
            timePickerFrom.ValueChanged += HandleChanged;
            timePickerTo.ValueChanged += HandleChanged;
            if (promptData.MinValue != null || hideNoValue)
            {
                chbNoValue.Visible = false;
            }

            List<DateTime> defaultValues = promptData.DefaultValues.Cast<PromptValue>().
                Where(v => !(v is RangePromptValue)).
                Select(v => TimePromptField.ToDateTime((TimeSpan)v.Value)).ToList();

            timePickerFrom.DefaultValues = timePickerTo.DefaultValues = defaultValues;
            timePickerFrom.CustomFormat = timePickerTo.CustomFormat = Thread.CurrentThread.CurrentCulture.DateTimeFormat.LongTimePattern;
            timePickerFrom.ShowUpDown = timePickerTo.ShowUpDown = true;
            timePickerFrom.DefaultsOnly = timePickerTo.DefaultsOnly = !promptData.Changeable;

            Value = promptData.Values;
            chbNoValue.Text = strings.Prompt_None_Long;
            ErrorProvider = new ErrorProvider(this);
            ControlForErrors = chbNoValue.Visible ? (Control)this.chbNoValue : (Control)this.timePickerTo;
        }

        /// <summary>
        /// Returns the chosen value for this panel. Never null: if null value, will return null wrapped in a SinglePromptValue.
        /// </summary>
        internal override PromptValue Value
        {
            get
            {
                RangePromptValue r = new RangePromptValue(PromptData.Time);
                r.Description = "";
                r.IncludeHigh = false; 
                r.IncludeLow = true;
                r.OnlyDescription = true;
                r.StartValue = new SinglePromptValue(TimePromptField.ToTimeSpan(timePickerFrom.Value), null, PromptData.Time);
                r.EndValue = new SinglePromptValue(TimePromptField.ToTimeSpan(timePickerTo.Value), null, PromptData.Time);
                return r;
            }
            set
            {
                if (value != null && value.Value is RangePromptValue)
                {
                    this.timePickerFrom.Value = TimePromptField.ToDateTime((TimeSpan)((RangePromptValue)value.Value).StartValue.Value);
                    this.timePickerTo.Value = TimePromptField.ToDateTime((TimeSpan)((RangePromptValue)value.Value).EndValue.Value);
                }
                else
                {
                    this.chbNoValue.Checked = true;
                }
            }
        }
    }
}
